using System;
using System.Collections;
using System.Data;
using System.Reflection;
using System.Text;

using gov.va.med.vbecs.Common;
using BrRules = gov.va.med.vbecs.Common.VbecsBrokenRules.Division;
using LblPrinterConfigRules = gov.va.med.vbecs.Common.VbecsBrokenRules.LabelPrinterConfig;
using TzConfigRules = gov.va.med.vbecs.Common.VbecsBrokenRules.TimeZoneConfig;
using DivTable = gov.va.med.vbecs.Common.VbecsTables.VamcDivision;
//using DstTable = gov.va.med.vbecs.Common.VbecsTables.DaylightSavings;

namespace gov.va.med.vbecs.BOL
{	
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Stanislav Antropov</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>7/7/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	/// <summary>
	/// This class contains minimal set of properties required to define division in the VBECS.     
	/// </summary>

	#endregion

	public class DivisionBase : DivisionDefinition
	{
		/// <summary>
		/// Minimum lock inactivity timeout in minutes.
		/// </summary>
		public const int MinLockInactivityTimeoutMin = 5;

		/// <summary>
		/// Maximum lock inactivity timeout in minutes.
		/// </summary>
		public const int MaxLockInactivityTimeoutMin = 15;

		private Guid _nationalFacilityGuid;
		private Guid _localSupplierGuid;

		private Guid _divisionGuid;
		private bool _divisionIsActive;
		private int _lockInactivityTimeoutMin;		
		private int _accessionAreaId;
		private bool _serviceTypeIndicator; // Indicates whether the VAMC division is a standard "full service" configured facility.
		
		private bool _divisionUsesLabelPrinter;
		private LabelPrinterConfig _labelPrinterConfig;
		private TimeZoneConfig _timeZoneConfig;		
		private string _printerName; // CR 2320

		private bool _inactivationPending;
		private bool _cachedIsBloodUnitNotInFinalStatus;
		private bool _bloodUnitNotInFinalStatusDataCached;
		private System.Data.DataTable _assocInstitutions;

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		/// <summary>
		/// Default constructor.
		/// </summary>
		protected DivisionBase() : base() 
		{
			this.SetInitialBrokenRules( BrRules.FormClassName, BrRules.RuleSets.BaseRuleSet );

			_divisionGuid = Guid.NewGuid();

			FacilityGuid = StNullConvert.GuidNull;
			_localSupplierGuid = StNullConvert.GuidNull;

			AccessionAreaId = StNullConvert.Int32Null;
			ServiceTypeIndicator = true;
			IsActive = false;
			LockInactivityTimeoutMin = LockManager.DefaultLockInactivityTimeoutMin;			
			UsesLabelPrinter = false;
			LabelPrinterConfig = null;
			TimeZoneConfig = null;
			PrinterName = string.Empty; // CR 2320

			IsNew = true;
			IsDirty = false;
			_inactivationPending = false;
			_cachedIsBloodUnitNotInFinalStatus = true; // this is a safety value
			_bloodUnitNotInFinalStatusDataCached = false;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6933"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6934"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///	
		///<Case type="0" testid ="6935"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6936"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="0" testid ="6937"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Constructor
		/// </summary>
		/// <param name="divisionDef"></param>
		public DivisionBase( DivisionDefinition divisionDef ) : this()
		{
			SetDivisionCode( divisionDef.DivisionCode );
			SetDivisionName( divisionDef.DivisionName );
		}

		/// <summary>
		/// Constructs an instance of the class loading required data from a given <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow">Source <see cref="DataRow"/> to load data from.</param>
		protected DivisionBase( DataRow dtRow ) : this()
		{			
			this.LoadFromDataRow( dtRow ); // DataRow is validated by the callee.
		}

		/// <summary>
		/// Loads data into this instance of the class from a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data from.</param>
		protected override void LoadFromDataRow( DataRow dtRow )
		{
			Common.Utility.RequireNonNullColumns( dtRow, GetRequiredColumns() );

			base.LoadFromDataRow( dtRow );

			_divisionGuid = StDbNullConvert.ToGuid( dtRow[ DivTable.DivisionGuid ] );
			
			FacilityGuid = StDbNullConvert.ToGuid( dtRow[ DivTable.FacilityGuid ] );
			_localSupplierGuid = StDbNullConvert.ToGuid( dtRow[ DivTable.LocalSupplierGuid ] );

			AccessionAreaId = StDbNullConvert.ToInt32( dtRow[ DivTable.AccessionAreaId ] );
			ServiceTypeIndicator = StDbNullConvert.ToBool( dtRow[ DivTable.ServiceTypeIndicator ], true );
			LockInactivityTimeoutMin = StDbNullConvert.ToInt32( dtRow[ DivTable.LockInactivityTimeoutMin ] );	

			UsesLabelPrinter = StDbNullConvert.ToBool( dtRow[ DivTable.LabelPrinterUseIndicator ], false );
			LabelPrinterConfig = _divisionUsesLabelPrinter ? new LabelPrinterConfig( dtRow ) : null;

			PrinterName = StDbNullConvert.ToString( dtRow[ DivTable.PrinterName ] ); // CR 2320

			IsActive = Common.Utility.GetRecordStatusCodeFromString( (string)dtRow[ DivTable.RecordStatusCode ].ToString() ) == Common.RecordStatusCode.Active;
			RowVersion = StDbNullConvert.ToRowversion( dtRow[ DivTable.RowVersion ] );

			IsActive = Utility.GetIsActiveStatusByStatusCode( dtRow[ DivTable.RecordStatusCode ] );
			RowVersion = StDbNullConvert.ToRowversion( dtRow[ DivTable.RowVersion ] );

			TimeZoneConfig = new TimeZoneConfig( dtRow );

			IsNew = false;
			IsDirty = false;
			_inactivationPending = false;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4852"> 
		///		<ExpectedInput>DataRow</ExpectedInput>
		///		<ExpectedOutput>Populated DataRow</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4853"> 
		///		<ExpectedInput>Invalid DataRow</ExpectedInput>
		///		<ExpectedOutput>ArgumentException</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Loads data from this instance of the class into a supplied <see cref="DataRow"/>.
		/// </summary>
		/// <param name="dtRow"><see cref="DataRow"/> to load data into.</param>
		/// <returns>Reference to provided <see cref="DataRow"/>.</returns>
		public override DataRow LoadDataRowFromThis( DataRow dtRow )
		{
			Common.Utility.RequireColumns( dtRow, GetRequiredColumns() );

			if( !IsValid )
				throw( new BusinessObjectException( StrRes.SysErrMsg.Common.UnableToSaveInvalidBusinessObject( typeof( DivisionBase ).Name ).ResString ) );	

			base.LoadDataRowFromThis( dtRow );

			dtRow[ DivTable.DivisionGuid ] = StDbNullConvert.From( _divisionGuid );
			dtRow[ DivTable.FacilityGuid ] = StDbNullConvert.From( _nationalFacilityGuid );
			dtRow[ DivTable.LocalSupplierGuid ] = StDbNullConvert.From( _localSupplierGuid );
			dtRow[ DivTable.AccessionAreaId ] = StDbNullConvert.From( _accessionAreaId );
			dtRow[ DivTable.ServiceTypeIndicator ] = StDbNullConvert.From( _serviceTypeIndicator );
			dtRow[ DivTable.LockInactivityTimeoutMin ] = StDbNullConvert.From( _lockInactivityTimeoutMin ); 
			dtRow[ DivTable.LabelPrinterUseIndicator ] = StDbNullConvert.From( _divisionUsesLabelPrinter ); // CR 2320
			dtRow[ DivTable.PrinterName ] = StDbNullConvert.From( _printerName );


			// BR_17.05
			if( this.IsNew ) 
				dtRow[ DivTable.InvoiceTemplateText ] = StDbNullConvert.From( StrRes.OtherMsg.MUC02.DefaultDivisionInvoiceText().ResString );

			dtRow[ DivTable.RecordStatusCode ] = Utility.GetRecordStatusCodeCharFromBoolean( IsActive );			
			dtRow[ DivTable.RowVersion ] = StDbNullConvert.From( RowVersion );

			if( _divisionUsesLabelPrinter )
				_labelPrinterConfig.LoadDataRowFromThis( dtRow );

			_timeZoneConfig.LoadDataRowFromThis( dtRow );

			return dtRow;
		}

		/// <summary>
		/// Gets list of columns required to import/export object via <see cref="DataRow"/>. 
		/// Note that only columns parsed immediately by the class are returned. 
		/// Columns used by aggregated objects are not included. 
		/// </summary>
		/// <returns>List of columns needed for import/export.</returns>
		protected virtual string[] GetRequiredColumns()
		{
			return new string[] {	DivTable.DivisionGuid, DivTable.FacilityGuid, DivTable.AccessionAreaId, 
									DivTable.ServiceTypeIndicator, DivTable.RecordStatusCode, 
									DivTable.LockInactivityTimeoutMin, DivTable.LabelPrinterUseIndicator };
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6274"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6275"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Reloads division base data from DB cancelling all changes. 
		/// </summary>
		public virtual void RefreshFromDb()
		{
			LoadFromDataRow( DAL.Division.GetDivision( this.DivisionCode ) );
		}

		/// <summary>
		/// GetTemplateDataTable
		/// </summary>
		/// <returns></returns>
		protected virtual DataTable GetTemplateDataTable()
		{
			DataTable _dt = new DataTable();

			_dt.Columns.Add( DivTable.DivisionCode, typeof(string) );
			_dt.Columns.Add( DivTable.DivisionName, typeof(string) );
			_dt.Columns.Add( DivTable.DivisionGuid, typeof(Guid) );
			_dt.Columns.Add( DivTable.FacilityGuid, typeof(Guid) );
			_dt.Columns.Add( DivTable.LocalSupplierGuid, typeof(Guid) );
			_dt.Columns.Add( DivTable.AccessionAreaId, typeof(int) );
			_dt.Columns.Add( DivTable.ServiceTypeIndicator, typeof(bool) );
			_dt.Columns.Add( DivTable.LabelPrinterUseIndicator, typeof(bool) );
			_dt.Columns.Add( DivTable.LabelPrinterComPortNumber, typeof(int) );
			_dt.Columns.Add( DivTable.LabelPrinterIPAddress, typeof(string) );
			_dt.Columns.Add( DivTable.LabelPrinterTcpPortNumber, typeof(int) );
			_dt.Columns.Add( DivTable.PrinterName, typeof(string) ); // CR 2320
			
			_dt.Columns.Add( DivTable.TimeZoneId, typeof(string) );
			_dt.Columns.Add( DivTable.DaylightSavingsTypeId, typeof(int) );
			_dt.Columns.Add( DivTable.DaylightSavingsStartDate, typeof(DateTime) );
			_dt.Columns.Add( DivTable.DaylightSavingsEndDate, typeof(DateTime) );
			_dt.Columns.Add( DivTable.LockInactivityTimeoutMin, typeof(int) );
			_dt.Columns.Add( DivTable.InvoiceTemplateText, typeof(string) );
			_dt.Columns.Add( DivTable.RecordStatusCode, typeof(char) );
			_dt.Columns.Add( DivTable.RowVersion, typeof(byte[]) );
			_dt.Columns.Add( DivTable.LastUpdateDate, typeof(DateTime) );
			_dt.Columns.Add( DivTable.LastUpdateUser, typeof(string) );

			return _dt;
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///		
		///<Case type="0" testid ="6276"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="6277"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Saves division base data to the database. Assumes that the operation is performed 
		/// in single user mode - no locks / rowversion checks is performed (MBR_1.09, MBR_1.10, MBR_1.11).
		/// </summary>
		/// <param name="updateFunction">Update function (calling unit ID).</param>
		public virtual void PersistDivisionBase( UpdateFunction updateFunction )
		{
			if( !IsDirty )
				return;

			// BR_6.03
			LocalSupplier facility = new LocalSupplier( FacilityDefinition.GetByGuid( this.FacilityGuid ).FdaRegistrationNumber, false );
			_localSupplierGuid = facility.LocalSupplierGuid;
			
			// MBR_2.11
			DataTable usersToUpdateCoreData;
			IList usersToUpdateDivisionRolesData;
		
			GetUsersToUpdateData( out usersToUpdateCoreData, out usersToUpdateDivisionRolesData );
			
			bool bypassUpdate = ( !facility.IsNew && facility.RecordStatusCode == RecordStatusCode.LocalEntry );

			if( facility.IsNew && facility.RecordStatusCode != RecordStatusCode.NationalStandard )
				throw( new BusinessObjectException( StrRes.SysErrMsg.MUC02.FacilityDefinitionNotFoundInNationalTable( facility.FDARegistrationNumber ).ResString ) );

			facility.RecordStatusCode = RecordStatusCode.LocalEntry;

			DataTable dtFacility = DAL.LocalSupplier.GetEmptyLocalSupplierTableSchema(!facility.IsNew);
			DataTable dtAddress = DAL.Address.GetEmptyAddressTableSchema(!facility.Address.IsNew);

			if (!bypassUpdate)
			{
				DataRow dr = facility.LoadDataRowFromThis(dtFacility.NewRow());
				dtFacility.Rows.Add(dr);

				dr = facility.LoadDataRowFromThis(dtAddress.NewRow());
				dtAddress.Rows.Add(dr);
			
				dtAddress = Common.Utility.AppendLastUpdateInformation(dtAddress, updateFunction);
				dtFacility = Common.Utility.AppendLastUpdateInformation(dtFacility, updateFunction);
			}

			DAL.Division.PersistDivisionBaseData( IsNew, ExportToDataTable(), dtFacility, dtAddress, facility.IsNew, facility.Address.IsNew, usersToUpdateCoreData, usersToUpdateDivisionRolesData, this._assocInstitutions, updateFunction );

			this.RefreshFromDb();
		}

		private void GetUsersToUpdateData( out DataTable usersToUpdateCoreData, out IList usersToUpdateDivisionRolesData )
		{
			IList usersToUpdateInCurrentDivision = GetListOfUsersToInactivateInCurrentDivision();
			
			VbecsUser.ExportUsersDataForSave( usersToUpdateInCurrentDivision, out usersToUpdateCoreData, out usersToUpdateDivisionRolesData );
		}

		private IList GetListOfUsersToInactivateInCurrentDivision()
		{
			IList usersToInactivateInCurrentDivision = IsInactivationPending ? VbecsUser.GetVbecsUsersWithRoleInGivenDivision( this.DivisionCode ) : new ArrayList( 1 );

			foreach( VbecsUser user in usersToInactivateInCurrentDivision )
				user.DivisionsAndRoles[ this.DivisionCode ].IsActive = false;
			
			return usersToInactivateInCurrentDivision;
		}
		
		/// <summary>
		/// Exports instance data to a <see cref="DataTable"/>.
		/// </summary>
		/// <returns><see cref="DataTable"/> containing instance data.</returns>
		protected virtual DataTable ExportToDataTable()
		{
			DataTable _dt = GetTemplateDataTable();
			DataRow _dr = _dt.NewRow();
			_dt.Rows.Add( _dr );

			LoadDataRowFromThis( _dr );

			return _dt;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4850"> 
		///		<ExpectedInput>None.</ExpectedInput>
		///		<ExpectedOutput>True for object loaded from DB, false for fresh object.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4851"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines whether the object is valid or not. 
		/// </summary>
		public override bool IsValid
		{
			get
			{
				return 
					base.IsValid && 
					( _labelPrinterConfig == null ? true : _labelPrinterConfig.IsValid ) &&
					( _timeZoneConfig == null ? true : _timeZoneConfig.IsValid );
			}
		}

		/// <summary>
		/// Invoked whenever object any of properties is changed.
		/// </summary>
		private void OnPropertyChanged()
		{
			IsDirty = true;
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/9/2005</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6932"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7449"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///	</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Division GUID.
		/// </summary>
		public virtual Guid DivisionGuid
		{
			get
			{
				return _divisionGuid;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6278"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6279"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// National facility GUID.
		/// </summary>
		public Guid FacilityGuid
		{
			get
			{
				return _nationalFacilityGuid;
			}
			set
			{
				// Checking for null GUID to be able to break the rule
				if( _nationalFacilityGuid == value && !StNullConvert.IsNull( _nationalFacilityGuid  ) )
					return;

				_nationalFacilityGuid = value;

				RuleBroken( BrRules.FacilityNameNotSet, StNullConvert.IsNull( _nationalFacilityGuid ) );

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4848"> 
		///		<ExpectedInput>int</ExpectedInput>
		///		<ExpectedOutput>int</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4849"> 
		///		<ExpectedInput>Invalid int</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Lock inactivity timeout in minutes.
		/// </summary>
		public virtual int LockInactivityTimeoutMin
		{
			get
			{
				return _lockInactivityTimeoutMin;
			}
			set
			{
				if( _lockInactivityTimeoutMin == value )
					return;					
				
				_lockInactivityTimeoutMin = value;

				RuleBroken( BrRules.LockInactivityTimeoutInvalid,					
					!RuleBroken( BrRules.LockInactivityTimeoutNotSet, StNullConvert.IsNull( _lockInactivityTimeoutMin ) ) 
					&& ( _lockInactivityTimeoutMin < MinLockInactivityTimeoutMin || _lockInactivityTimeoutMin > MaxLockInactivityTimeoutMin ) );

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6280"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6281"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Blood bank accession area. 
		/// </summary>
		public virtual int AccessionAreaId 
		{
			get
			{
				return _accessionAreaId;
			}
			set
			{
				if( _accessionAreaId == value )
					return;

				_accessionAreaId = value;

				RuleBroken( BrRules.AccessionAreaNotSet, StNullConvert.IsNull( _accessionAreaId ) );

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/23/2002</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="286"> 
		///		<ExpectedInput>Division object</ExpectedInput>
		///		<ExpectedOutput>Service type indicator of division object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="764"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets or sets value indicating whether the VAMC division is a standard "full service" configured facility.
		/// A true value indicates Full Service, a false value indicates transfusion only.
		/// </summary>
		public bool ServiceTypeIndicator
		{
			get 
			{ 
				return _serviceTypeIndicator;
			}
			set 
			{	
				if( _serviceTypeIndicator == value )
					return;
				
				_serviceTypeIndicator = value;				
				OnPropertyChanged();
			}
		}

		// TODO: add unit tests
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/31/2002</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="6282"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7450"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// Gets string representation of division service type indicator
		/// </summary>
		public string ServiceType
		{
			get
			{
				return ( this._serviceTypeIndicator ? StrRes.OtherMsg.Common.DivisionServiceTypeFull() : StrRes.OtherMsg.Common.DivisionServiceTypeTransfusionOnly() ).ResString;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4846"> 
		///		<ExpectedInput>True/False values</ExpectedInput>
		///		<ExpectedOutput>Corresponding passed in value.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4847"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates whether a division is active or not. 
		/// </summary>
		public virtual bool IsActive
		{
			get
			{
				return _divisionIsActive;
			}
			set
			{
				if( _divisionIsActive == value )
					return;

				_inactivationPending = _divisionIsActive && !IsNew && !value;
				
				_divisionIsActive = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>VHA</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/23/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8508"> 
		///		<ExpectedInput>String</ExpectedInput>
		///		<ExpectedOutput>String</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="8509"> 
		///		<ExpectedInput>Empty String</ExpectedInput>
		///		<ExpectedOutput>Broken Rule</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates the name of the default report printer. 
		/// CR 2320
		/// </summary>
		public virtual string PrinterName
		{
			get
			{
				return _printerName;
			}
			set
			{
				if( _printerName == value )
					return;
		
				_printerName = value;

				RuleBroken( BrRules.DefaultPrinterNotSet,( _printerName == null  || _printerName == string.Empty ));

				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/13/2005</CreationDate>
		///<TestCases>
		///		
		///<Case type="0" testid ="6284"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7451"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		Specifies if inactivation is pending for the division 
		///		(active division is marked is inactive but not yet stored to DB). 
		/// </summary>
		public virtual bool IsInactivationPending
		{
			get
			{
				return _inactivationPending;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6285"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6286"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		Indicates whether the current DivisionBase object is new.
		/// </summary>
		public override bool IsNew
		{
			get
			{
				return base.IsNew;
			}
			set
			{
				if( base.IsNew == value )
					return;

				if( value && _inactivationPending )
					_inactivationPending = false;

				base.IsNew = value;
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4844"> 
		///		<ExpectedInput>True/False values</ExpectedInput>
		///		<ExpectedOutput>Corresponding passed in value.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4845"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Indicates if division uses label printer. 
		/// </summary>
		public virtual bool UsesLabelPrinter
		{
			get
			{
				return _divisionUsesLabelPrinter;
			}
			set
			{
				if( _divisionUsesLabelPrinter == value )
					return;					

				_divisionUsesLabelPrinter = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4842"> 
		///		<ExpectedInput>LabelPrinterConfig</ExpectedInput>
		///		<ExpectedOutput>LabelPrinterConfig</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4843"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Configuration for printer used in division for printing bag labels.  
		/// </summary>
		public virtual LabelPrinterConfig LabelPrinterConfig
		{
			get
			{
				return _labelPrinterConfig;
			}
			set
			{
				if( _labelPrinterConfig == value )
					return;

				_labelPrinterConfig = value;
				OnPropertyChanged();
			}
		}

	


		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4854"> 
		///		<ExpectedInput>TimeZoneConfig</ExpectedInput>
		///		<ExpectedOutput>TimeZoneConfig</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4855"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Division time zone and daylight savings configuration.
		/// </summary>
		public virtual TimeZoneConfig TimeZoneConfig
		{
			get
			{
				return _timeZoneConfig;
			}
			set
			{
				if( _timeZoneConfig == value )
					return;
		
				_timeZoneConfig = value;
				OnPropertyChanged();
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6287"> 
		///		<ExpectedInput>NONE</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6288"> 
		///		<ExpectedInput>Valid altering of the LabelPrinterConfig attribute</ExpectedInput>
		///		<ExpectedOutput>true</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6289"> 
		///		<ExpectedInput>Valid altering of the TimeZoneConfig attribute</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="6292"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		///		IsDirty inidcates that the current DivisionBase object
		///		has been altered in some way but has not been save to the
		///		database yet.
		/// </summary>
		public override bool IsDirty
		{
			get
			{
				return	base.IsDirty || 
					( this.LabelPrinterConfig != null && this.LabelPrinterConfig.IsDirty ) ||
					( this.TimeZoneConfig != null && this.TimeZoneConfig.IsDirty );
			}
			set
			{
				base.IsDirty = value;
			}
		}

		///<Developers>
		///	<Developer>Rob Heiberger</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>1/14/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6293"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="6294"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// GetBrokenRulesMessage
		/// </summary>
		/// <returns></returns>
		public override string GetBrokenRulesMessage( string formClassName )
		{
			if( formClassName == null )
				throw( new ArgumentNullException( "formClassName" ) );

			BrokenRulesMessageBuilder _mb = new BrokenRulesMessageBuilder();
			_mb.AppendMessage( base.GetBrokenRulesMessage( formClassName ) );
			
			if( LabelPrinterConfig != null )
				_mb.AppendMessage( LabelPrinterConfig.GetBrokenRulesMessage( LblPrinterConfigRules.FormClassName ) );

			if( TimeZoneConfig != null )
				_mb.AppendMessage( TimeZoneConfig.GetBrokenRulesMessage( TzConfigRules.FormClassName ) );

			return _mb.ToString();
		}

		// TODO: add unit tests
		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/22/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="6931"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>boolean</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7452"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns a boolean telling if units exist in the division which are not in a final status.
		/// BR_9.09
		/// </summary>
		/// <returns></returns>
		public bool IsBloodUnitNotInFinalStatus()
		{
			return DAL.Division.IsBloodUnitNotInFinalStatus( this.DivisionCode );
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/8/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7453"> 
		///		<ExpectedInput>Valid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7454"> 
		///		<ExpectedInput>Invalid Parameters</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Same as <see cref="IsBloodUnitNotInFinalStatus"/> but executed only once - 
		/// when the value is requested for the first time. Use <see cref="IsBloodUnitNotInFinalStatus"/>
		/// to get up to date information. This is used in MUC02 to speed up access since 
		/// no blood units may be changed in division while MUC02 is running (VBECS won't start). 
		/// </summary>
		/// <returns>Boolean telling if unit not in final status exists in the division.</returns>
		public bool CachedIsBloodUnitNotInFinalStatus()
		{
			if( !_bloodUnitNotInFinalStatusDataCached )
			{
				_cachedIsBloodUnitNotInFinalStatus = IsBloodUnitNotInFinalStatus();
				_bloodUnitNotInFinalStatusDataCached = true;
			}

			return _cachedIsBloodUnitNotInFinalStatus;
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/7/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4838"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Non-zero list of divisions.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4839"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Gets list of all divisions available in the VBECS.
		/// </summary>
		/// <returns>List of all VBECS divisions.</returns>
		public static IList GetBaseInfoForAllVbecsDivisions()
		{
			DataTable _dt = DAL.Division.GetDivisions( false );
			ArrayList _list = new ArrayList( _dt.Rows.Count );

			foreach( DataRow _dr in _dt.Rows )
				_list.Add( new DivisionBase( _dr ) );

			return _list;
		}

	

		/// <summary>
		/// GUID for local supplier associated with the division. 
		/// It changes after facility GUID is set and <see cref="PersistDivisionBase"/> called.
		/// </summary>
		protected Guid VolatileLocalSupplierGuid
		{
			get
			{
				return _localSupplierGuid;
			}
		}
		///<Developers>
		///<Developer>Carrie Van Stedum</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/27/2007</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8499"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>A list of all associated institutions for a particular division base instance.</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="8500"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///	
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		///
		/// <summary>
		/// Gets/Sets a datatable populated with all the institutions associated with a VBECS division, CR 2316, DR 2723 
		/// </summary>
		public System.Data.DataTable AssociatedInstitutions
		{
			get	
			{
				_assocInstitutions = DAL.Division.GetAssociatedInstitutionsByDivision(this.DivisionCode);
				return _assocInstitutions;
			}
			set
			{
                _assocInstitutions = value;
				OnPropertyChanged();
			}
		}
	}
}